/*------------------------------------------------------------------------------------
   Name: TMS-Osc Histo 3TF.mq4
   Copyright 2010, Xaphod, http://forexwhiz.appspot.com
   
   Description: 3 time frame tape chart of the 'Trading Made Simple' System Oscillator. 
                Similar to the TDI RSI Signal Line and Trade Signal Line.
	          
   Change log: 
       2011-06-17. Xaphod, v1.00
          - Fixed Alerts.
          - Fixed? missing htf buffer data on start.
       2011-05-22. Xaphod, v0.91
          - Added Auto Time-frame setting
       2011-05-22. Xaphod, v0.90
          - First Release 
-------------------------------------------------------------------------------------*/
// Indicator properties
#property copyright "Copyright  2010, Xaphod"
#property link      "http://forexwhiz.appspot.com"

#property indicator_separate_window
#property indicator_buffers 6
#property indicator_color1 LimeGreen
#property indicator_color2 Red
#property indicator_color3 LimeGreen
#property indicator_color4 Red
#property indicator_color5 LimeGreen
#property indicator_color6 Red
#property indicator_minimum 0
#property indicator_maximum 1.5

//#include <xDebug.mqh>

// Constant definitions
#define INDICATOR_NAME "TMS-Osc Histo 3TF"
#define INDICATOR_VERSION "1.00"
#define INDICATOR_DISP_VC "1.00, 2011, Xaphod"
#define BAR_CHAR 110


// Indicator parameters
extern string    Indi.Version=INDICATOR_DISP_VC;
extern string    TimeFrame.Settings="";
extern int       TimeFrame.TF1=5;           // Time-Frame 1
extern int       TimeFrame.TF2=15;          // Time-Frame 1
extern int       TimeFrame.TF3=60;          // Time-Frame 1
extern bool      TimeFrame.Auto=True;       // Automatically select higher TFs. M15->H1, M30->H1.
extern bool      TimeFrame.Repaint=True;    // Repaint the display for all time-frames higher than the charts time-frame
extern string    RSI.Settings="";
extern int       RSI.Period=13;             // RSI Period
extern int       RSI.Price=PRICE_CLOSE;     // RSI Price Type: CLOSE=0, OPEN=1, HIGH=2, LOW=3, MEDIAN=4, TYPICAL=5, WEIGHTED=6
extern string    RSISignal.Settings="";
extern int       RSISignal.Period=2;        // RSI Signal MA-Filter Period
extern int       RSISignal.Mode=MODE_SMA;   // MA-Filter Method: SMA=0, EMA=1, SMMA=2, LWMA=4
extern string    TradeSignal.Settings="";
extern int       TradeSignal.Period=7;      // Trade Signal MA-Filter Period
extern int       TradeSignal.Mode=MODE_SMA; // MA-Filter Method: SMA=0, EMA=1, SMMA=2, LWMA=4
extern string    Alert.Settings="";
extern bool      Alert.Enable=True;        // Enable Alerts
extern bool      Alert.OnBarClose=True;     // Alert only when an open bar closes
extern bool      Alert.Popup=True;         // Popup window & sound on alert
extern bool      Alert.Email=False;         // Send email on alert
extern string    Alert.Subject="Default";   // Email Subject. "Default" will result in a preconfigured subject.
extern string    Display.Settings="";
extern color     Display.LabelColor=White;  // Color of Histogram Id labels
extern bool      Display.LowToHighTF=true;  // Set to show histogram with the lowest TF on the top. 
extern string    Display.Instance="";       // Set to a unique value to use multiple copies of the indicator on the same chart. 

// Global module varables
double gadTF1Up[];
double gadTF1Dn[];
double gadTF2Up[];
double gadTF2Dn[];
double gadTF3Up[];
double gadTF3Dn[];
double gadRSI[];
double gadRMAFast[];
double gadRMASlow[];
// Globals
int giRepaintBars;
string gsIndicatorName;
bool gbInit;
double gdGap[3];
bool gbDataError=False;

//-----------------------------------------------------------------------------
// function: init()
// Description: Custom indicator initialization function.
//-----------------------------------------------------------------------------
int init() {
  string sTFList;
  int iMaxTF;
  
  
  // Init indicator buffers
  SetIndexStyle(0,DRAW_ARROW);
  SetIndexBuffer(0,gadTF1Up);
  SetIndexStyle(1,DRAW_ARROW);
  SetIndexBuffer(1,gadTF1Dn);
  SetIndexStyle(2,DRAW_ARROW);
  SetIndexBuffer(2,gadTF2Up);
  SetIndexStyle(3,DRAW_ARROW);
  SetIndexBuffer(3,gadTF2Dn);
  SetIndexStyle(4,DRAW_ARROW);
  SetIndexBuffer(4,gadTF3Up);
  SetIndexStyle(5,DRAW_ARROW);
  SetIndexBuffer(5,gadTF3Dn);
  for (int i = 0; i < indicator_buffers; i++) {
    SetIndexLabel(i,NULL);
    SetIndexEmptyValue(i,0.0);
    SetIndexArrow(i,BAR_CHAR);
  }
  
  // Set Timeframe
  if (TimeFrame.Auto) {
    TimeFrame.TF1=Period();
    TimeFrame.TF2=NextHigherTF(TimeFrame.TF1);
    TimeFrame.TF3=NextHigherTF(TimeFrame.TF2);
  }
  
  // name and stuff
  if (Display.LowToHighTF)
    sTFList=TF2Str(TimeFrame.TF1)+","+TF2Str(TimeFrame.TF2)+","+TF2Str(TimeFrame.TF3);
  else
    sTFList=TF2Str(TimeFrame.TF3)+","+TF2Str(TimeFrame.TF2)+","+TF2Str(TimeFrame.TF1);  
  gsIndicatorName=INDICATOR_NAME+Display.Instance+"("+sTFList+")";
  IndicatorShortName(gsIndicatorName);
  IndicatorDigits(0);
  
  // Set number of bars to redraw
  iMaxTF=TimeFrame.TF1;
  if (TimeFrame.TF2>iMaxTF) iMaxTF=TimeFrame.TF2;
  if (TimeFrame.TF3>iMaxTF) iMaxTF=TimeFrame.TF3;
  giRepaintBars=iMaxTF/Period()+2; 
  
  gbInit=True;
  return(0);
}

//-----------------------------------------------------------------------------
// function: deinit()
// Description: Custom indicator deinitialization function.
//-----------------------------------------------------------------------------
int deinit() {
  int i;
  
  // Clear text objects
  for( i=ObjectsTotal()-1; i>-1; i--)
    if (StringFind(ObjectName(i),INDICATOR_NAME)>=0)  ObjectDelete(ObjectName(i));
  return (0);
}


//-----------------------------------------------------------------------------
// function: start()
// Description: Custom indicator iteration function.
//-----------------------------------------------------------------------------
int start() {
  int iNewBars;
  int iCountedBars; 
  int i;
  double gdRMAFast,gdRMASlow;
  double dGap[3];
  static datetime tCurBar;
  static datetime tAlertBar;
  
  if (gbInit) {
    // Set histogram positions
    if (Display.LowToHighTF) { 
      dGap[0]=1.2;
      dGap[1]=0.7;
      dGap[2]=0.2;
    }
    else { 
      dGap[0]=0.2;
      dGap[1]=0.7;
      dGap[2]=1.2;
    }  
  }
  
  // Get unprocessed ticks
  iCountedBars=IndicatorCounted();
  if(iCountedBars < 0) return (-1); 
  if(iCountedBars>0) iCountedBars--;
  if (gbDataError) {    
    iNewBars=Bars-1;
    gbDataError=false;
  }
  else {
    iNewBars=Bars-iCountedBars;
  }
  
  // Set bars to redraw
  if (TimeFrame.Repaint && iNewBars<giRepaintBars)
    iNewBars=giRepaintBars;

  // Calc RSI MAs for timeframe 1
  CalcRSIMA(TimeFrame.TF1);
  for(i=iNewBars; i>=0; i--) {
    // Draw tape chart
    gdRMAFast=gadRMAFast[iBarShift(Symbol(), TimeFrame.TF1, Time[i])];
    gdRMASlow=gadRMASlow[iBarShift(Symbol(), TimeFrame.TF1, Time[i])];
     
    // Check RSI MA order and update display
    gadTF1Up[i] = EMPTY;
    gadTF1Dn[i] = EMPTY;            
    if (gdRMAFast>gdRMASlow)
      gadTF1Up[i] = dGap[0];
    else if (gdRMAFast<gdRMASlow)
      gadTF1Dn[i] = dGap[0];
  }
  
  // Calc RSI MAs for timeframe 2
  CalcRSIMA(TimeFrame.TF2);
  for(i=iNewBars; i>=0; i--) {
    // Draw tape chart
    gdRMAFast=gadRMAFast[iBarShift(Symbol(), TimeFrame.TF2, Time[i])];
    gdRMASlow=gadRMASlow[iBarShift(Symbol(), TimeFrame.TF2, Time[i])];
       
    // Check MA order and update display
    gadTF2Up[i] = EMPTY;
    gadTF2Dn[i] = EMPTY;            
    if (gdRMAFast>gdRMASlow)
      gadTF2Up[i] = dGap[1];
    else if (gdRMAFast<gdRMASlow)
      gadTF2Dn[i] = dGap[1];
  }
  
  // Calc RSI MAs for timeframe 3
  CalcRSIMA(TimeFrame.TF3);
  for(i=iNewBars; i>=0; i--) {
    // Draw tape chart
    gdRMAFast=gadRMAFast[iBarShift(Symbol(), TimeFrame.TF3, Time[i])];
    gdRMASlow=gadRMASlow[iBarShift(Symbol(), TimeFrame.TF3, Time[i])];
      
    // Check MA order and update display
    gadTF3Up[i] = EMPTY;
    gadTF3Dn[i] = EMPTY;            
    if (gdRMAFast>gdRMASlow)
      gadTF3Up[i] = dGap[2];
    else if (gdRMAFast<gdRMASlow)
      gadTF3Dn[i] = dGap[2];    
  }

  // Alert on the close of the current bar
  if (Alert.Enable && Alert.OnBarClose && tAlertBar<iTime(Symbol(),TimeFrame.TF1,iBarShift(Symbol(), TimeFrame.TF1, Time[0])) && !gbInit) {
    tAlertBar=iTime(Symbol(),TimeFrame.TF1,iBarShift(Symbol(), TimeFrame.TF1, Time[0]));
    if (TrendUp(1) && !TrendUp(2)) 
      DoAlert(Symbol()+", "+TF2Str(Period())+": Trend Up."); 
    else if (TrendDown(1) && !TrendDown(2))
      DoAlert(Symbol()+", "+TF2Str(Period())+": Trend Down.");
  }
  
  // Alert while the current bar is open
  if (Alert.Enable && !Alert.OnBarClose && tAlertBar<iBarShift(Symbol(), TimeFrame.TF1, Time[0]) && !gbInit) {
    tAlertBar=Time[iBarShift(Symbol(), TimeFrame.TF1, Time[0])];
    if (TrendUp(0) && !TrendUp(1))
      DoAlert(Symbol() + ", " + TF2Str(Period())+ ": Trend Up.");
    else if (TrendDown(0) && !TrendDown(1))
      DoAlert(Symbol()+", "+ TF2Str(Period()) + ": Trend Down.");
  }
  
  // Tasks to execute on bar close
  if (tCurBar<Time[0]) {
    tCurBar=Time[0];
    
    // Write/Update bar labels
    Writelabel(TF2Str(TimeFrame.TF1),dGap[0]+0.2);
    Writelabel(TF2Str(TimeFrame.TF2),dGap[1]+0.2);
    Writelabel(TF2Str(TimeFrame.TF3),dGap[2]+0.2);
    
    // Clear the init flag   
    if (gbInit) gbInit=False;
  }
  
  return(0);
}
//+------------------------------------------------------------------+


//-----------------------------------------------------------------------------
// function: CalcRSIMA()
// Description: Calculate the RSI MAs
//-----------------------------------------------------------------------------
void CalcRSIMA(int iTimeFrame) {
  int i;
  int iCalcBars;
  int iError=0;
  datetime tTimeArray[];

  // Init arrays 
  GetLastError();
  ArrayCopySeries(tTimeArray,MODE_TIME,Symbol(),iTimeFrame); 
  iError=GetLastError();
  if (iError!=0) gbDataError=True;    
  iCalcBars=ArraySize(tTimeArray);
  ArraySetAsSeries(gadRSI,false);
  ArraySetAsSeries(gadRMAFast,false);
  ArraySetAsSeries(gadRMASlow,false);
  ArrayResize(gadRSI,iCalcBars);
  ArrayResize(gadRMAFast,iCalcBars);
  ArrayResize(gadRMASlow,iCalcBars);  
  ArraySetAsSeries(gadRSI,true);
  ArraySetAsSeries(gadRMAFast,true);
  ArraySetAsSeries(gadRMASlow,true);
  // Calc HTF data
  for (i=iCalcBars; i>=0; i--) {  
    gadRSI[i]=iRSI(Symbol(),iTimeFrame,RSI.Period,RSI.Price,i);      
  }
  for (i=iCalcBars; i>=0; i--) {
    gadRMAFast[i]=iMAOnArray(gadRSI,0,RSISignal.Period,0,RSISignal.Mode,i);
    gadRMASlow[i]=iMAOnArray(gadRSI,0,TradeSignal.Period,0,TradeSignal.Mode,i);
  }
  return(0);
}

//-----------------------------------------------------------------------------
// function: DoAlert()
// Description: Signal the popup and email alerts
//-----------------------------------------------------------------------------
void DoAlert(string sAlertMsg) {
  //Popup Alert 
  if (Alert.Popup) 
    Alert(sAlertMsg, " ", INDICATOR_NAME);      
  //Email Alert
  if (Alert.Email) {
    if (Alert.Subject=="Default") Alert.Subject=sAlertMsg;
    SendMail( Alert.Subject, "MT4 Alert!\n"+INDICATOR_NAME+"\n" + TimeToStr(Time[0],TIME_DATE|TIME_SECONDS )+"\n"+sAlertMsg);          
  }
}


//-----------------------------------------------------------------------------
// function: TrendUp()
// Description: Return true if the trend is up. Fast MAs > Slow MAs.
//              Ie the moving averages are in ascending order. 
//-----------------------------------------------------------------------------
bool TrendUp(int i) {
  if (gadTF1Up[i]!=EMPTY && gadTF2Up[i]!=EMPTY && gadTF3Up[i]!=EMPTY)
    return(True);
  else
    return(False);
}


//-----------------------------------------------------------------------------
// function: TrendDown()
// Description: Return true if the trend is down. Fast MAs < Slow MAs.
//              Ie the moving averages are in decending order
//-----------------------------------------------------------------------------
bool TrendDown(int i) {
  if (gadTF1Dn[i]!=EMPTY && gadTF2Dn[i]!=EMPTY && gadTF3Dn[i]!=EMPTY)
    return(True);
  else
    return(False);
}


//-----------------------------------------------------------------------------
// function: Writelabel()
// Description: Write a label for a bar
//-----------------------------------------------------------------------------
int Writelabel(string sLabel,double dPrice) {
  string sObjId;
  sObjId=INDICATOR_NAME+sLabel;
  if(ObjectFind(sObjId) < 0) 
    ObjectCreate(sObjId, OBJ_TEXT, WindowFind(gsIndicatorName), Time[0]+Period()*60*3, dPrice);
  ObjectSetText(sObjId, sLabel, 7, "Arial", Display.LabelColor);
  ObjectMove(sObjId,0,Time[0]+Period()*60*3, dPrice);
  return(0);
}


//-----------------------------------------------------------------------------
// function: TF2Str()
// Description: Convert time-frame to a string
//-----------------------------------------------------------------------------
string TF2Str(int iPeriod) {
  switch(iPeriod) {
    case 0 : return("M"+Period());
    case PERIOD_M1: return("M1");
    case PERIOD_M5: return("M5");
    case PERIOD_M15: return("M15");
    case PERIOD_M30: return("M30");
    case PERIOD_H1: return("H1");
    case PERIOD_H4: return("H4");
    case PERIOD_D1: return("D1");
    case PERIOD_W1: return("W1");
    case PERIOD_MN1: return("MN1");
    default: return("M"+iPeriod);
  }
  return(0);
}

//-----------------------------------------------------------------------------
// function: NextHigherTF()
// Description: Select the next higher time-frame. 
//              Note: M15 and M30 both select H1 as next higher TF. 
//-----------------------------------------------------------------------------
int NextHigherTF(int iPeriod) {
  if (iPeriod==0) iPeriod=Period();
  switch(iPeriod) {
    case PERIOD_M1: return(PERIOD_M5);
    case PERIOD_M5: return(PERIOD_M15);
    case PERIOD_M15: return(PERIOD_H1);
    case PERIOD_M30: return(PERIOD_H1);
    case PERIOD_H1: return(PERIOD_H4);
    case PERIOD_H4: return(PERIOD_D1);
    case PERIOD_D1: return(PERIOD_W1);
    case PERIOD_W1: return(PERIOD_MN1);
    case PERIOD_MN1: return(PERIOD_MN1);
    default: return(Period());
  }
}